﻿using System;
using System.Web.Mvc;
using System.Linq;
using MVC_App.Models;
using MVC_App.Validation;

namespace MVC_App.Controllers
{
	public partial class HomeController : Controller
	{
		[HttpGet]
		public ActionResult EditOrder()
		{
			return View(BuildDummyBestellung());
		}

		static BestellungForm BuildDummyBestellung()
		{
			return new BestellungForm
					{
						Lieferadresse = new LieferadresseForm(),
						Kreditkarte = new KreditkarteForm(),
						Pizza = new PizzaForm
											{
												Größe = PizzaGrößen.Mittel,
											}
					};
		}

		[HttpPost]
		public ActionResult SubmitOrder(BestellungForm bestellung)
		{
			DateTime? lieferzeitpunkt = ValidateLieferzeitpunkt(bestellung);

			ZahlungModel zahlung = ValidateZahlung(bestellung);

			GetränkModel getränk = null;
			if (bestellung.HasGetränk)
				getränk = ValidateGetränk(bestellung);

			PizzaModel pizza = ValidatePizza(bestellung);

			DessertModel dessert = null;
			if (bestellung.HasDessert)
				dessert = ValidateDessert(bestellung);

			if (ModelState.IsValid
				&& lieferzeitpunkt != null
				&& zahlung != null
				&& !(bestellung.HasGetränk && getränk == null)
				&& pizza != null
				&& !(bestellung.HasDessert && dessert == null))
			{
				var orderModel = new OrderModel(lieferzeitpunkt.Value, zahlung, getränk, pizza, dessert);
				return View("Success", orderModel);
			}

			return View("EditOrder", bestellung);
		}

		private DateTime? ValidateLieferzeitpunkt(BestellungForm bestellung)
		{
			Time time;
			switch (bestellung.LieferzeitpunktArt)
			{
				case Lieferzeitpunkte.Sofort:
					return DateTime.Now;

				case Lieferzeitpunkte.Heute:
					time = ValidateUhrzeit(bestellung);
					if (time == null)
						return null;

					return DateTime.Now.SetTime(time);

				case Lieferzeitpunkte.Am:
					time = ValidateUhrzeit(bestellung);
					var date = ValidateDatum(bestellung);

					if (time == null || date == null)
						return null;

					return date.Value.SetTime(time);

				default:
					return null;
			}
		}

		private Time ValidateUhrzeit(BestellungForm bestellung)
		{
			if (bestellung.Lieferzeitpunkt.Uhrzeit != null)
			{
				var result = Time.Build(bestellung.Lieferzeitpunkt.Uhrzeit);
				if (result != null)
					return result;
			}

			ModelState.AddModelError("Lieferzeitpunkt.Uhrzeit", "Bitte geben Sie eine gültige Uhrzeit für die Lieferung an.");
			return null;
		}

		private DateTime? ValidateDatum(BestellungForm bestellung)
		{
			if (bestellung.Lieferzeitpunkt.Datum != null)
			{
				var result = DateTimeEx.Build(bestellung.Lieferzeitpunkt.Datum);
				if (result != null)
					return result;
			}

			ModelState.AddModelError("Lieferzeitpunkt.Datum", "Bitte geben Sie ein gültiges Datum für die Lieferung an.");
			return null;
		}

		private ZahlungModel ValidateZahlung(BestellungForm bestellung)
		{
			switch (bestellung.Zahlmodus)
			{
				case Zahlmodi.Haustür:
					return new HaustürZahlungModel();
				case Zahlmodi.Kreditkarte:
					return ValidateKreditkarteZahlung(bestellung);
				default:
					ModelState.AddModelError("Zahlmodus", "Bitte wählen Sie eine Zahlungmethode aus");
					return null;
			}
		}

		private static DateTime BuildDateTime(int monat, int jahr)
		{
			try
			{
				return new DateTime(jahr, monat, 1);
			}
			catch (Exception)
			{
				throw new ValidationException("Ungültige Datumsangabe");
			}
		}

		private GetränkModel ValidateGetränk(BestellungForm bestellung)
		{
			if (bestellung.Getränk.Name == null)
				ModelState.AddModelError("Getränk.Name", "Bitte wählen Sie ein gültiges Getränk aus");

			int größe = -1;
			if (bestellung.Getränk.Flaschengröße == null
				|| !int.TryParse(bestellung.Getränk.Flaschengröße, out größe))
				ModelState.AddModelError("Getränk.Flaschengröße", "Bitte wählen Sie eine gültige Größe aus");

			if (bestellung.Getränk.Name != null && größe != -1)
			{
				var getränkSorte = GetränkSorteRepository.Find(bestellung.Getränk.Name);
				if (getränkSorte == null)
					ModelState.AddModelError("Getränk.Name", "Bitte wählen Sie ein gültiges Getränk aus");
				else
				{
					if (!getränkSorte.VerfügbareVolumen.Contains(größe))
						ModelState.AddModelError("Getränk.Flaschengröße",
						                         "Das von Ihnen gewählte Getränk ist nicht in dieser Größe erhältlich");
					else
						return new GetränkModel(getränkSorte, bestellung.Getränk.Flaschengröße, bestellung.Getränk.Gekühlt);
				}
			}
			return null;
		}

		private PizzaModel ValidatePizza(BestellungForm bestellung)
		{
			bool error = false;
			if (!bestellung.Pizza.Größe.HasValue)
			{
				ModelState.AddModelError("Pizza.Größe", "Bitte wählen Sie eine gültige Pizzagröße aus");
				error = true;
			}
			if (string.IsNullOrWhiteSpace(bestellung.Pizza.Sorte))
			{
				ModelState.AddModelError("Pizza.Sorte", "Bitte wählen Sie eine gültige Pizzasorte aus");
				error = true;
			}

			if (error)
				return null;

			if (!PizzaRepository.Sorten.Contains(bestellung.Pizza.Sorte))
			{
				ModelState.AddModelError("Pizza.Sorte", "Bitte wählen Sie eine gültige Pizzasorte aus");
				return null;
			}

			return new PizzaModel(bestellung.Pizza.Sorte, bestellung.Pizza.Größe.Value);
		}

		private DessertModel ValidateDessert(BestellungForm bestellung)
		{
			// Wenn das DessertForm keine Nutzdaten hat wird es garnicht erst erstellt.
			// Dann macht es hier Bumm!
			if (!bestellung.Dessert.Name.HasValue)
			{
				ModelState.AddModelError("Dessert.Name", "Bitte wählen Sie ein gültiges Dessert aus");
				return null;
			}
			return new DessertModel(bestellung.Dessert.Name.Value);
		}
	}
}
